/**********************************************************************************
*   this file is part of c2h
*   Copyright (C)2005 Bruce Park ( jongsuknim@naver.com )
*
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation; either
*   version 2 of the License, or (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************************/
#include "general.h"	

#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <stdio.h>
#include <ctype.h>

#include <fcntl.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/stat.h>

#include "debug.h"
#include "routines.h"

#ifndef TMPDIR
# define TMPDIR "/tmp"
#endif

#ifndef S_ISREG
# if defined (S_IFREG) && ! defined (AMIGA)
#  define S_ISREG(mode)	    ((mode) & S_IFREG)
# else
#  define S_ISREG(mode)	    TRUE	/* assume regular file */
# endif
#endif

#ifndef S_ISLNK
# ifdef S_IFLNK
#  define S_ISLNK(mode)	    (((mode) & S_IFMT) == S_IFLNK)
# else
#  define S_ISLNK(mode)	    FALSE	/* assume no soft links */
# endif
#endif

#ifndef S_ISDIR
# ifdef S_IFDIR
#  define S_ISDIR(mode)	    (((mode) & S_IFMT) == S_IFDIR)
# else
#  define S_ISDIR(mode)	    FALSE	/* assume no soft links */
# endif
#endif

#ifndef S_IFMT
# define S_IFMT 0
#endif

#ifndef S_IXUSR
# define S_IXUSR 0
#endif
#ifndef S_IXGRP
# define S_IXGRP 0
#endif
#ifndef S_IXOTH
# define S_IXOTH 0
#endif

#ifndef S_IRUSR
# define S_IRUSR 0400
#endif
#ifndef S_IWUSR
# define S_IWUSR 0200
#endif

#ifndef S_ISUID
# define S_ISUID 0
#endif

/*  Hack for rediculous practice of Microsoft Visual C++.
 */
#if defined (WIN32)
# if defined (_MSC_VER)
#  define stat    _stat
#  define getcwd  _getcwd
#  define currentdrive() (_getdrive() + 'A' - 1)
#  define PATH_MAX  _MAX_PATH
# elif defined (__BORLANDC__)
#  define PATH_MAX  MAXPATH
#  define currentdrive() (getdisk() + 'A')
# elif defined (DJGPP)
#  define currentdrive() (getdisk() + 'A')
# else
#  define currentdrive() 'C'
# endif
#endif

#ifndef PATH_MAX
# define PATH_MAX 256
#endif

/*
 *  Miscellaneous macros
 */
#define selected(var,feature)	(((int)(var) & (int)(feature)) == (int)feature)

/*
*   DATA DEFINITIONS
*/
char *CurrentDirectory;

extern void error (const errorSelection selection,
		   const char *const format, ...)
{
    va_list ap;

    va_start (ap, format);
    fprintf (errout, "%s", selected (selection, WARNING) ? "Warning: " : "");
    vfprintf (errout, format, ap);
    if (selected (selection, PERROR))
	fprintf (errout, " : %s\n", strerror (errno));
    fputs ("\n", errout);
    va_end (ap);
    if (selected (selection, FATAL))
	exit (1);
}

/*
 *  Memory allocation functions
 */

extern void *eMalloc (const size_t size)
{
    void *buffer = malloc (size);

    if (buffer == NULL)
	error (FATAL, "out of memory");

    return buffer;
}

extern void *eCalloc (const size_t count, const size_t size)
{
    void *buffer = calloc (count, size);

    if (buffer == NULL)
	error (FATAL, "out of memory");

    return buffer;
}

extern void *eRealloc (void *const ptr, const size_t size)
{
    void *buffer;
    if (ptr == NULL)
	buffer = eMalloc (size);
    else
    {
	buffer = realloc (ptr, size);
	if (buffer == NULL)
	    error (FATAL, "out of memory");
    }
    return buffer;
}

extern void Free (void *const ptr)
{
    Assert (ptr != NULL);
    free (ptr);
}

typedef struct Alloc_list{
    const char *file;
    int line;
    const char *addr;
    int size;
    struct Alloc_list *next;
}alloc_list;

alloc_list *head = NULL;

static int total_alloc_size = 0;

static void insert_alloc_list( const char *addr, int size, const char *file , int line )
{
    alloc_list *list;
    list = (alloc_list*)malloc( sizeof(alloc_list));
    if( list == NULL )
	error( FATAL, "out of memory");

    list->file = file;
    list->line = line;
    list->addr = addr;
    list->size = size;
    list->next = head;
    head = list;

    total_alloc_size += size;
}

static void remove_alloc_list( const char *addr ,const char *file, int line)
{
    alloc_list *tmp,*prev;
    int size;
    for( tmp = head , prev=NULL; tmp != NULL ; tmp = tmp->next )
    {
	if( tmp->addr == addr ){
	    if( prev == NULL ){
		size = tmp->size;
		prev = head;
		head = head->next;
		free(prev);
	    }
	    else{
		size = tmp->size;
		prev->next = tmp->next;
		free(tmp);
	    }
	    total_alloc_size -= size;
	    return ;
	}
	prev = tmp;
    }
    fprintf(stderr,"MEM_LIST error: %s  %d\n", file, line);
}

extern void *dMalloc (const size_t size , const char *file, int line)
{
    void *buffer = malloc (size);

    if (buffer == NULL)
	error (FATAL, "out of memory");

    insert_alloc_list( buffer, size,file, line );
    return buffer;
}
extern void *dCalloc (const size_t count, const size_t size , const char *file , int line)
{
    void *buffer = calloc (count, size);

    if (buffer == NULL)
	error (FATAL, "out of memory");

    insert_alloc_list( buffer, size, file, line );
    return buffer;
}
extern void *dRealloc (void *const ptr, const size_t size , const char *file, int line)
{
    void *buffer;
    if (ptr == NULL)
	buffer = eMalloc (size);
    else
    {
	buffer = realloc (ptr, size);
	if (buffer == NULL)
	    error (FATAL, "out of memory");
    }
    insert_alloc_list( buffer, size, file, line );
    return buffer;
}
extern void dFree (void *const ptr , const char *file, int line)
{
    //printf("%s %d\n", file, line );
    Assert (ptr != NULL);
    remove_alloc_list(ptr , file, line);
    free (ptr);
}

extern void showUnfreeMem()
{
    alloc_list *tmp;
    int num =0;
    //FILE *fd = fopen("alloc_list", "w");
    for( tmp = head; tmp != NULL ; tmp = tmp->next )
    {
	fprintf( stderr, "[%15s %7d] size:%d   addr:%10X\n", tmp->file, tmp->line, tmp->size, (unsigned int)(tmp->addr) );
	num++;
    }
    fprintf( stderr, "total num : %d    total alloc size: %d\n", num, total_alloc_size );
}

/*
 *  String manipulation functions
 */

extern int struppercmp (const char *s1, const char *s2)
{
    int result;
    do
    {
	result = toupper ((int) *s1) - toupper ((int) *s2);
    } while (result == 0  &&  *s1++ != '\0'  &&  *s2++ != '\0');
    return result;
}

extern int strnuppercmp (const char *s1, const char *s2, size_t n)
{
    int result;
    do
    {
	result = toupper ((int) *s1) - toupper ((int) *s2);
    } while (result == 0  &&  --n > 0  &&  *s1++ != '\0'  &&  *s2++ != '\0');
    return result;
}

extern char* eStrdup (const char* str)
{
    char* result = xMalloc (strlen (str) + 1, char);
    strcpy (result, str);
    return result;
}


/*
 * File system functions
 */

extern void setCurrentDirectory (void)
{
    char* buf;
    if (CurrentDirectory == NULL)
	CurrentDirectory = xMalloc ((size_t) (PATH_MAX + 1), char);
    buf = getcwd (CurrentDirectory, PATH_MAX);
    if (buf == NULL)
	perror ("");
    if (CurrentDirectory [strlen (CurrentDirectory) - (size_t) 1] !=
	    PATH_SEPARATOR)
    {
	sprintf (CurrentDirectory + strlen (CurrentDirectory), "%c",
		OUTPUT_PATH_SEPARATOR);
    }
}


/* For caching of stat() calls */
static fileStatus *sStat (const char *const fileName)
{
    struct stat status;
    static fileStatus file;

    if (file.name != NULL)
	eFree (file.name);
    file.name = eStrdup (fileName);
    if (lstat (file.name, &status) != 0)
	file.exists = FALSE;
    else
    {
	file.isSymbolicLink = (boolean) S_ISLNK (status.st_mode);
	if (file.isSymbolicLink  &&  stat (file.name, &status) != 0)
	    file.exists = FALSE;
	else
	{
	    file.exists = TRUE;
	    file.isDirectory = (boolean) S_ISDIR (status.st_mode);
	    file.isNormalFile = (boolean) (S_ISREG (status.st_mode));
	    file.isExecutable = (boolean) ((status.st_mode &
		(S_IXUSR | S_IXGRP | S_IXOTH)) != 0);
	    file.isSetuid = (boolean) ((status.st_mode & S_ISUID) != 0);
	    file.size = status.st_size;
	}
    }
    return &file;
}

extern void eStatFree( fileStatus *stat)
{
    Assert( stat != NULL);
    Assert( stat->name != NULL );
    eFree( stat->name );
    eFree( stat );
}

extern fileStatus *eStat (const char *const fileName)
{
    fileStatus *dest, *sour;
    dest = xMalloc( 1, fileStatus );
    sour = sStat( fileName );
    memcpy( dest, sour , sizeof(fileStatus));
    dest->name = eStrdup( sour->name );
    return dest;
}

extern boolean doesFileExist (const char *const fileName)
{
    boolean ret;
    fileStatus *status = sStat (fileName);
    if ( status->exists )
	ret = TRUE;
    else 
	ret = FALSE;
    
    return status->exists;
}

/*
 *  Pathname manipulation (O/S dependent!!!)
 */

static boolean isPathSeparator (const int c)
{
    boolean result;
    result = (boolean) (c == PATH_SEPARATOR);
    return result;
}

extern boolean isSameFile (const char *const name1, const char *const name2)
{
    boolean result = FALSE;
    struct stat stat1, stat2;

    if (stat (name1, &stat1) == 0  &&  stat (name2, &stat2) == 0)
	result = (boolean) (stat1.st_ino == stat2.st_ino);

    return result;
}

extern const char *baseFilename (const char *const filePath)
{
    const char *tail = strrchr (filePath, PATH_SEPARATOR);
    if (tail == NULL)
	tail = filePath;
    else
	++tail;			/* step past last delimiter */

    return tail;
}

extern const char *fileExtension (const char *const fileName)
{
    const char *extension;
    const char *pDelimiter = NULL;
    const char *const base = baseFilename (fileName);
    if (pDelimiter == NULL)
        pDelimiter = strrchr (base, '.');

    if (pDelimiter == NULL)
	extension = "";
    else
	extension = pDelimiter + 1;	/* skip to first char of extension */

    return extension;
}

extern boolean isAbsolutePath (const char *const path)
{
    boolean result = FALSE;
    result = isPathSeparator (path [0]);
    return result;
}

static char* concat (const char *s1, const char *s2, const char *s3)
{
  int len1 = strlen (s1), len2 = strlen (s2), len3 = strlen (s3);
  char *result = xMalloc (len1 + len2 + len3 + 1, char);

  strcpy (result, s1);
  strcpy (result + len1, s2);
  strcpy (result + len1 + len2, s3);
  result [len1 + len2 + len3] = '\0';

  return result;
}

/* Return a newly allocated string containing the absolute file name of FILE
 * given CWD (which should end with a slash).
 * Routine adapted from Gnu etags.
 */
extern char* absoluteFilename (const char *file)
{
    char *slashp, *cp;
    char *res = NULL;
    if (isAbsolutePath (file))
    {
	res = eStrdup (file);
    }
    else
	res = concat (CurrentDirectory, file, "");

    /* Delete the "/dirname/.." and "/." substrings. */
    slashp = strchr (res, PATH_SEPARATOR);
    while (slashp != NULL  &&  slashp [0] != '\0')
    {
	if (slashp[1] == '.')
	{
	    if (slashp [2] == '.' &&
		(slashp [3] == PATH_SEPARATOR || slashp [3] == '\0'))
	    {
		cp = slashp;
		do
		    cp--;
		while (cp >= res  &&  ! isAbsolutePath (cp));
		if (cp < res)
		    cp = slashp;/* the absolute name begins with "/.." */
		strcpy (cp, slashp + 3);
		slashp = cp;
		continue;
	    }
	    else if (slashp [2] == PATH_SEPARATOR  ||  slashp [2] == '\0')
	    {
		strcpy (slashp, slashp + 2);
		continue;
	    }
	}
	slashp = strchr (slashp + 1, PATH_SEPARATOR);
    }

    if (res [0] == '\0')
	return eStrdup ("/");
    else
    {
	return res;
    }
}

extern char* absoluteDirname (const char *file)
{
    char *slashp, *res;
    char save;
    slashp = strrchr (file, PATH_SEPARATOR);
    if (slashp == NULL)
	res = eStrdup (CurrentDirectory);
    else
    {
	save = slashp [1];
	slashp [1] = '\0';
	res = absoluteFilename (file);
	slashp [1] = save;
    }
    return res;
}


extern char* relativeFilename (const char *file, const char *dir)
{
    const char *fp, *dp;
    char *absfile, *res, *absdir;
    int i;

    if( dir[0] == '/' ){
	absdir = (char*)dir;
    }else{
	absdir = absoluteFilename( dir );
    }
    /* Find the common root of file and dir (with a trailing slash). */
    absfile = absoluteFilename (file);

    fp = absfile;
    dp = absdir;
    while (*fp++ == *dp++)
	continue;
    fp--;
    dp--;			/* back to the first differing char */
    do
    {				/* look at the equal chars until path sep */
	if (fp == absfile)
	    return absfile;	/* first char differs, give up */
	fp--;
	dp--;
    } while (*fp != PATH_SEPARATOR);

    /* Build a sequence of "../" strings for the resulting relative file name.
     */
    i = 0;
    while ((dp = strchr (dp + 1, PATH_SEPARATOR)) != NULL)
	i += 1;
    res = xMalloc (3 * i + strlen (fp + 1) + 1, char);
    res [0] = '\0';
    while (i-- > 0)
	strcat (res, "../");

    /* Add the file name relative to the common root of file and dir. */
    strcat (res, fp + 1);
    free (absfile);

    return res;
}

extern void createRecursiveDir( const char *const target )
{
    char buf[BUF_MAX];
    int i=1;

    while(TRUE)
    {
	for( ; i< BUF_MAX; i++ )
	{
	    if( target[i] == '/' ){
		strncpy( buf, target, i );
		buf[i++] = '\0';
		break;
	    }
	    else if( target[i] == '\0' ){
		if( !doesFileExist(target)){
#if 0
		    fprintf(stderr,"make dir %s/n", target );
#endif
		    if( mkdir(target, DIR_MODE )< 0 )
			error( FATAL, "create dir error %s :\n\t\t%s   %d ",
				buf, __FILE__, __LINE__);
		}
		return;
	    }
	}
	if( !doesFileExist( buf ))
	    if( mkdir(buf, DIR_MODE )< 0 )
		error( FATAL, "create dir error %s :\n\t\t%s   %d ",
			buf, __FILE__, __LINE__);
    }
}

/*
    The first block or so of the file is examined for odd characters 
    such as strange control codes or characters with the high bit set. 
    If too many strange characters (>30%) are found, it's a -B file, 
    otherwise it's a -T file. Also, any file containing null 
    in the first block is considered a binary file.
*/
extern boolean is_binary( const char *fname )
{
    char buf[512];
    FILE *fd;
    size_t size;
    int i;
    int b;
    fd = fopen( fname , "r");
    if( fd == NULL )
	error( FATAL, "file open error%s  %s %d\n", fname, __FILE__, __LINE__ );

    size = fread( buf, 1, 512, fd );
    fclose(fd);
    if ( size == 0 )
	return TRUE;
    
    for( i=0, b=0; i< size; i++ )
    {
	if( buf[i] == 0 )
	    return TRUE;
	if( !isprint((int)buf[i]))
	    b++;
    }
    b = b * 100;
    if( b / size > 30 )
	return TRUE;
    else 
	return FALSE;
}

extern int ungets  ( const char *str , FILE *fd)
{
    int i,len ;
    len = strlen(str);
    for( i = len-1; i >= 0 ; i-- )
	if( ungetc( (int)str[i] , fd ) < 0 )
	    return -1;
    return len;
}

/*
   ݵ / տ ־ Ѵ.
 */
extern const char *get_base_dir( const char *const dir_name )
{
    int len;
    int i;
    len = strlen( dir_name );
    for( i = len -2 ; i >= 0 ; i-- )
    {
	if( dir_name[i] == '/')
	    return &dir_name[i+1];
    }
    Assert(FALSE);
    return NULL;
}
/*
    get_base_dir  ٸ տ /  ȴ. 
 */
extern const char *get_base_file( const char *const dir_name )
{
    int len;
    int i;
    len = strlen( dir_name );
    for( i = len -2 ; i >= 0 ; i-- )
    {
	if( dir_name[i] == '/')
	    return &dir_name[i+1];
    }
    return dir_name;
}


extern const char *remove_html_ext( const char *const file_name )
{
    static char *ret = NULL;
    int size = strlen(file_name) - 5;
    if( ret != NULL )
	eFree( ret );
    
    ret = xMalloc( size +1, char );
    strncpy( ret, file_name , size );
    ret[size] = '\0';
    return ret;
}

extern char *get_common_dir(const char *const file1, const char *const file2)
{
    int i=0;
    char buf[BUF_MAX];
    while( file1[i] != 0 || file2[i] != 0 )
    {
	if( file1[i] == file2[i] )
	    buf[i]  = file1[i];
	else
	    break;
	i++;
    }
    buf[i]  = 0;
    return eStrdup( buf );
}


extern char *get_dir_name( const char *const fname )
{
    char *dir;
    //int len = strlen( fname );
    int i,j;
    for( i = strlen( fname) - 2; i > 0  ; i-- )
	if( fname[i] == '/' )
	    break;

    if( i == 0 ){
	dir = xMalloc( 1, char);
	dir[0] = '\0';
    }else{
	i++;
	dir = xMalloc( i+2, char);
	for( j = 0 ; j < i; j++ )
	    dir[j] = fname[j];
	dir[j] = 0;
    }
    //fprintf(stderr, "%s\n", fname ); 
    //fprintf(stderr, "%s\n", dir ); 
    
    return dir;
}

